<?php
/*--------------------------------------------------------------------------------------------------
    PaddingGroupOption.php 2020-07-16
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2020 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------------------------------------*/

namespace Gambio\StyleEdit\Core\Components\PaddingGroup\Entities;

use Gambio\StyleEdit\Core\Components\Option\Entities\Option;
use Gambio\StyleEdit\Core\Components\TextBox\Entities\TextBox;
use Gambio\StyleEdit\Core\Options\Entities\AbstractComponentGroupOption;
use Gambio\StyleEdit\Core\SingletonPrototype;

/**
 * Class PaddingGroupOption
 * @package Gambio\StyleEdit\Core\Components\PaddingGroup\Entities
 */
class PaddingGroupOption extends AbstractComponentGroupOption
{
    /**
     * @var TextBox
     */
    protected $paddingTop;
    
    /**
     * @var TextBox
     */
    protected $paddingRight;
    
    /**
     * @var TextBox
     */
    protected $paddingBottom;
    
    /**
     * @var TextBox
     */
    protected $paddingLeft;
    
    
    /**
     * PaddingGroupOption constructor.
     */
    public function __construct()
    {
        parent::__construct();
        
        $this->paddingTop    = SingletonPrototype::instance()->get('TextOption');
        $this->paddingRight  = SingletonPrototype::instance()->get('TextOption');
        $this->paddingBottom = SingletonPrototype::instance()->get('TextOption');
        $this->paddingLeft   = SingletonPrototype::instance()->get('TextOption');
    }
    
    
    /**
     * @return TextBox
     */
    public function paddingTop(): TextBox
    {
        return $this->paddingTop;
    }
    
    
    /**
     * @return TextBox
     */
    public function paddingRight(): TextBox
    {
        return $this->paddingRight;
    }
    
    
    /**
     * @return TextBox
     */
    public function paddingBottom(): TextBox
    {
        return $this->paddingBottom;
    }
    
    
    /**
     * @return TextBox
     */
    public function paddingLeft(): TextBox
    {
        return $this->paddingLeft;
    }
    
    
    /**
     * Specify data which should be serialized to JSON
     * @link  https://php.net/manual/en/jsonserializable.jsonserialize.php
     * @return mixed data which can be serialized by <b>json_encode</b>,
     * which is a value of any type other than a resource.
     * @since 5.4.0
     */
    public function jsonSerialize()
    {
        $result          = new \stdClass;
        $result->id      = $this->id();
        $result->type    = 'padding';
        $result->label   = $this->label();
    
        if ($this->pageNamespace()) {
            $result->pageNamespace = $this->pageNamespace();
        }
        
        $result->items   = [
            'top'    => $this->paddingTop()->value(),
            'right'  => $this->paddingRight()->value(),
            'bottom' => $this->paddingBottom()->value(),
            'left'   => $this->paddingLeft()->value(),
        ];
        $result->default = [
            'top'    => $this->paddingTop()->defaultValue(),
            'right'  => $this->paddingRight()->defaultValue(),
            'bottom' => $this->paddingBottom()->defaultValue(),
            'left'   => $this->paddingLeft()->defaultValue(),
        ];
        
        return $result;
    }
    
    
    /**
     * @param $value
     *
     * @return boolean
     */
    protected function isValid($value): bool
    {
        return true;
    }
    
    
    /**
     * @param $value
     *
     * @return mixed
     */
    protected function parseValue($value)
    {
        return $value;
    }
    
    
    public function __clone()
    {
        parent::__clone(); // TODO: Change the autogenerated stub
        $this->paddingTop    = clone $this->paddingTop;
        $this->paddingRight  = clone $this->paddingRight;
        $this->paddingBottom = clone $this->paddingBottom;
        $this->paddingLeft   = clone $this->paddingLeft;
    }
    
    
    /**
     * @param $object
     *
     * @throws \Exception
     */
    public function initializeFromJsonObject($object): void
    {
        if (empty($object->items)) {
            $object->items = $object->value->distances ?? null;
            unset($object->value);
        }
        
        if (isset($object->id)) {
            $this->id = $object->id;
        }
        
        if (isset($object->label)) {
            $this->label = $object->label;
        }

        if (isset($object->pageNamespace)) {
            $this->pageNamespace = $object->pageNamespace;
        }

        $settings = ['id' => $this->id() . '-top', 'type' => $this->type() ? : null];
        if (isset($object->default, $object->default->top)) {
            $settings['default'] = $object->default->top;
        }
        if (isset($object->items, $object->items->top)) {
            $settings['value'] = $object->items->top;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->paddingTop->initializeFromJsonObject((object)$settings);
        
        $settings = ['id' => $this->id() . '-right', 'type' => $this->type() ? : null];
        if (isset($object->default, $object->default->right)) {
            $settings['default'] = $object->default->right;
        }
        if (isset($object->items, $object->items->right)) {
            $settings['value'] = $object->items->right;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->paddingRight->initializeFromJsonObject((object)$settings);
        
        $settings = ['id' => $this->id() . '-bottom', 'type' => $this->type() ? : null];
        if (isset($object->default, $object->default->bottom)) {
            $settings['default'] = $object->default->bottom;
        }
        if (isset($object->items, $object->items->bottom)) {
            $settings['value'] = $object->items->bottom;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->paddingBottom->initializeFromJsonObject((object)$settings);
        
        $settings = ['id' => $this->id() . '-left', 'type' => $this->type() ? : null];
        if (isset($object->default, $object->default->left)) {
            $settings['default'] = $object->default->left;
        }
        if (isset($object->items, $object->items->left)) {
            $settings['value'] = $object->items->left;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->paddingLeft->initializeFromJsonObject((object)$settings);
    }
    
    
    /**
     * @return mixed
     */
    public function getGroupOptions()
    {
        return [
            $this->paddingTop(),
            $this->paddingRight(),
            $this->paddingBottom(),
            $this->paddingLeft()
        ];
    }
    
    
    /**
     * @return string
     */
    public function type(): ?string
    {
        return 'padding';
    }
}